#import "NSXMutableDictionary.h"


void testMutableDictionary()
{
    // create objects to be keys and values in the dictionary
    NSString *immutableKey1     = @"key 1";
    NSString *immutableKey2     = @"key 2";
    NSString *immutableKey3     = @"key 3";
    NSString *immutableValue1   = @"value 1";
    NSString *immutableValue2   = @"value 2";
    NSString *immutableValue3   = @"value 3";
    
    NSMutableString *mutableKey     = [[NSMutableString alloc] initWithString:@"mutableKey"];
    NSMutableString *mutableValue   =  [[NSMutableString alloc] initWithString:@"mutableValue"];
    
    
    // create a dictionary
    NSXMutableDictionary *dict1 = [[NSXMutableDictionary alloc] init];
    
    // create a pointer for error
    NSError *error;
    
    
    
    // NOTE: In addition to assertions below, we will also check how you manage memory in your code.
    //       Therefore, be careful with your use of "retain", "release", "autorelease"
    
    
    
    // add an object into the dictionary no error should occur
    [dict1 setObject:immutableValue1 forKey:immutableKey1 error:&error];
    NSCAssert(error == nil, @"error object should be nil here");
    
    
    
    // add the second object to the dictionary, the count should now be 2
    [dict1 setObject:immutableValue2 forKey:immutableKey2 error:&error];
    NSCAssert([dict1 count] == 2, @"incorrect object count");

    
    
    // retrieving object by a key should return the correct value
    NSObject *retrievedValue1 = [dict1 objectForKey:immutableKey1];
    NSCAssert(retrievedValue1 == immutableValue1, @"incorrect value retrieved for immutableKey1");

    
    
    // add a nil to the dictionary, we should have an error with code 1
    error = nil;
    [dict1 setObject:nil forKey:immutableKey3 error:&error];
    NSCAssert(error != nil, @"there should be an error because of nil value");
    NSCAssert(error.code == 1, @"the error code should be 1");
    // NOTE: You are prohibited from using convenient constructor [NSError errorWithDomain:code:userInfo:].
    // NOTE: NSXMutableDictionary creates and owns the error object. Be careful of the reference count of the object you are returning here.
    
    
    
    // using nil as a key also generate an error with code 2
    error = nil;
    [dict1 setObject:immutableValue2 forKey:nil error:&error];
    NSCAssert(error != nil, @"there should be an error because of nil key");
    NSCAssert(error.code == 2, @"the error code should be 2");
    
    
    // you should copy the key; otherwise it might be modified outside without your awareness
    [dict1 setObject:immutableValue3 forKey:mutableKey error:&error];
    NSObject *retrievedValue3 = [dict1 objectForKey:mutableKey];
    NSCAssert(retrievedValue3 != nil, @"we should get a value associated with the original mutable key here");
    
    [mutableKey appendString:@"mod"];
    NSObject *retrievedValue4 = [dict1 objectForKey:mutableKey];
    NSCAssert(retrievedValue4 == nil, @"no object is associated with the specified key, this should return nil");
    
    
    // releasing the original key object should not affect the key stored in the dictionary
    [mutableKey release];
    mutableKey = nil;
    NSMutableString *retrievedKey3 = (NSMutableString *)[dict1 lastKeySuccessfullySet];
    NSCAssert([retrievedKey3 isEqualToString:@"mutableKey"], @"the last key should stay the same");
    // NOTE: mind the retain count of the return value of "lastKeySuccessfullySet" method
    
    
    // changing the object that is a value should be reflected when we retrieved in the object
    error = nil;
    [dict1 setObject:mutableValue forKey:immutableKey3 error:&error];
    [mutableValue appendString:@"mod"];
    NSMutableString *retrievedValue6 = (NSMutableString *)[dict1 objectForKey:immutableKey3];
    NSCAssert(error == nil, @"value in the dictionary should be mutable");
    NSCAssert([retrievedValue6 isEqualToString:@"mutableValuemod"], @"the retrieved value should be changed according to the change in mutableValue variable");
    NSCAssert(retrievedValue6 == mutableValue, @"the address of the retrieved value must still be the same");
    
    
    
    // the dictionary should retain the ownership of the value objects
    [mutableValue release];
    mutableValue = nil;
    NSMutableString *retrievedValue7 = (NSMutableString *)[dict1 objectForKey:immutableKey3];
    NSCAssert([retrievedValue7 isEqualToString:@"mutableValuemod"], @"the retrieved value should still be alive");
    
    
    
    // your dictionary should retain the ownership beyond the autorelease pool
    @autoreleasepool
    {
        NSString *autoreleasedValue1 = @"autoreleasedValue";
        [dict1 setObject:autoreleasedValue1 forKey:immutableKey1 error:&error];
    }
    NSString *retrievedValue8 = (NSString *)[dict1 objectForKey:immutableKey1];
    NSCAssert([retrievedValue8 isEqualToString:@"autoreleasedValue"], @"the value should still be alive regardless of the autoreleasing of the original object");
    // NOTE the "retrievedValue8" should point to the same memory address as "autoreleasedValue1".
    
    
    // removing object should release the object previously held by the dictionary
    @autoreleasepool
    {
        NSString *autoreleasedValue2 = @"autoreleasedValue";
        [dict1 setObject:immutableKey1 forKey:autoreleasedValue2 error:&error];
    }
    NSObject *retrievedValue9 = [dict1 objectForKey:immutableKey1];
    [dict1 removeObjectForKey:immutableKey1];
    // NOTE: at this point retrievedValue9 should point to an invalid memory address
    
    
    // deleting the dictionary
    [dict1 release];
    // NOTE: your dictionary should give up the ownership of all keys and values here
}